% Optimising k-Wave Performance Example
%
% This example demonstrates how to increase the computational performance
% of k-Wave using optional input parameters and data casting.  
%
% author: Bradley Treeby
% date: 15th July 2009
% last update: 22nd July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% change scale to 2 to reproduce the higher resolution figures given in the
% help menu
scale = 1;

% load the initial pressure distribution from an image and scale
p0_magnitude = 2;
p0 = p0_magnitude*loadImage('EXAMPLE_source_two.bmp');

% assign the grid size and create the computational grid
Nx = 256*scale;           % number of pixels in the x (column) direction
Nz = 256*scale;           % number of pixels in the z (row) direction
x = 10e-3;          % grid size [m]
z = 10e-3;          % grid size [m]
dx = x/Nx;          % pixel width [m]
dz = z/Nz;          % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% resize the input image to the desired number of pixels
p0 = resize(p0, Nx, Nz);

% define the properties of the propagation medium
c = 1500;           % [m/s]
rho = 1000;         % [kg/m^3]

% define a centered Cartesian circular sensor
sensor_radius = 4.5e-3;     % [m]
num_sensor_points = 100;
sensor_mask = makeCartCircle(sensor_radius, num_sensor_points);

% create the time array
[t_array dt] = makeTime(kgrid, c);

% set the input options
input_args = {'CartInterp', 'nearest', 'PlotSim', false};

% run the simulation
kspaceFirstOrder2D(p0, kgrid, c, rho, t_array, sensor_mask, input_args{:});

% set the input options to include data casting
input_args = {'CartInterp', 'nearest', 'PlotSim', false, 'DataCast','single'};

% run the simulation again
kspaceFirstOrder2D(p0, kgrid, c, rho, t_array, sensor_mask, input_args{:});